/*****************************************************************************/

/*
 *
 *   Copyright (c) 2002, Smart Link Ltd.
 *   All rights reserved.
 *
 *   Redistribution and use in source and binary forms, with or without
 *   modification, are permitted provided that the following conditions
 *   are met:
 *
 *       1. Redistributions of source code must retain the above copyright
 *          notice, this list of conditions and the following disclaimer.
 *       2. Redistributions in binary form must reproduce the above
 *          copyright notice, this list of conditions and the following
 *          disclaimer in the documentation and/or other materials provided
 *          with the distribution.
 *       3. Neither the name of the Smart Link Ltd. nor the names of its
 *          contributors may be used to endorse or promote products derived
 *          from this software without specific prior written permission.
 *
 *   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *   OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

/*
 *
 *      sltty.c  --  TTY driver for Smart Link Modem.
 *
 *      Authors: Alex Raer, Sasha K (sashak@smlink.com)
 *
 *
 */

/*****************************************************************************/

#include <linux/version.h>
#include <linux/module.h>
#include <linux/mm.h>
#include <linux/tty.h>
#include <linux/tty_flip.h>
#include <linux/termios.h>
#include <linux/fs.h>
#include <linux/serial.h>
#include <linux/sched.h>
#include <linux/tqueue.h>
#include <linux/spinlock.h>
#include <linux/ctype.h>
#include <asm/uaccess.h>

#include <linux_compat.h>

/*
 * TODO, FIX:
 * - race on module unloading during tty_hangup()
 *
 */

#define SLTTY_ERR(fmt...)  printk(KERN_ERR "sltty: " fmt)
#if MODEM_DEBUG
#include <modem_debug.h>
#define printk dprintf
//#define SLTTY_PARANOIA_CHECK 1
//#define SLTTY_PARANOIA_PRINT 1
#define SLTTY_DBG(fmt...)  printk(KERN_DEBUG fmt)
#else
#define SLTTY_DBG(fmt...)
#endif

/*
 *	These are now numbers
 */

#define SLTTY_NMAJOR	212
#define	SLTTY_MAGIC	(('M' << 8) | 'T')

#define MIN(a, b) ( (a) < (b) ? (a) : (b) )


struct sltty_port {
	unsigned short magic;
	spinlock_t spin_lock;
	unsigned initialized;
	unsigned closing;
	int count;
	int blocked_open;
	unsigned short status;
	void *modem;
	struct tty_struct *tty;
	wait_queue_head_t close_wait;
	wait_queue_head_t open_wait;
	struct tq_struct modem_task;
	unsigned char *xmit_buf;
	int xmit_head;
	int xmit_tail;
	int xmit_cnt;
	int xmit_size;
};


/*
 *   Prototypes.
 *
 */

extern void *modem_create(struct sltty_port *port);
extern int modem_delete(void *modem);
extern int modem_open(void *modem);
extern int modem_close(void *modem);
extern int modem_hangup(void *modem);
extern int modem_run(void *modem);

/* static data */

static int sltty_refcount = 0;

static struct tty_driver sltty;
static struct tty_struct *sltty_table = NULL;
static struct termios *sltty_termios = NULL;
static struct termios *sltty_termios_locked = NULL;

static struct sltty_port sltty_port;

static int  sltty_write_room(struct tty_struct *tty);
static void sltty_write_wakeup(struct sltty_port *port);

static unsigned char *tmp_buf = 0;
static DECLARE_MUTEX(tmp_buf_sem);


/*
 *  misc routines ...
 *
 */

extern inline int sltty_paranoia_check(struct sltty_port const *port,
				       const char *routine)
{
#ifdef SLTTY_PARANOIA_CHECK
	static const char *badport = KERN_ERR
		"sltty: ERR: bad port(%p) addrs in %s.\n";
	static const char *badmagic = KERN_ERR
		"sltty: ERR: bad port(%p) magic in %s.\n";
	static const char *badtty = KERN_ERR
		"sltty: ERR: bad port(%p) tty in %s.\n";
#if 0
	static char *paranoia_check = "sltty: %s...\n";
	SLTTY_DBG(paranoia_check, routine);
#endif
	if (port != &sltty_port) {
		printk(badport, port, routine);
		return -1;
	}
	if (port->magic != SLTTY_MAGIC) {
		printk(badmagic,  port, routine);
		return -1;
	}
	if (!port->tty) {
		printk(badtty, port, routine);
		return -1;
	}
#endif
	return 0;
}


/*
 *
 * public funcs for Port driver
 *
 */

void sltty_update_status(struct sltty_port *port, unsigned status)
{
	unsigned long flags;
	unsigned old_status;
	SLTTY_DBG("sltty_update_status...\n");
	if(sltty_paranoia_check(port,"sltty_update_status"))
		return;

	spin_lock_irqsave(&port->spin_lock,flags);
	old_status   = port->status;
	port->status = status | (old_status&(TIOCM_RTS|TIOCM_DTR));
	spin_unlock_irqrestore(&port->spin_lock,flags);

	/* DCD changed */
	if ((status&TIOCM_CD) != (old_status&TIOCM_CD)) {
		if (status&TIOCM_CD) { /* carrier has been detected */
			SLTTY_DBG("sltty: carrier detected.\n");
			if (port->tty && !C_CLOCAL(port->tty)) {
				SLTTY_DBG("sltty: wakeup open.\n");
				wake_up_interruptible(&port->open_wait);
			}
		}
		else {                  /* carrier has been lost */
			SLTTY_DBG("sltty: carrier has been lost.\n");
			if (port->tty && !C_CLOCAL(port->tty)) {
				SLTTY_DBG("sltty: hangup tty.\n");
				tty_hangup(port->tty);
			}
		}
	}
	/* CTS/RTS flow is handled automatically by modem */
}

int sltty_read_count(struct sltty_port *port)
{
	unsigned long flags;
	int ret;
	if (sltty_paranoia_check(port,"sltty_read_count"))
		return -1;
	if (test_bit(TTY_THROTTLED,&port->tty->flags))
		return TTY_FLIPBUF_SIZE;
	spin_lock_irqsave(&port->spin_lock,flags);
	ret = port->tty->flip.count;
	spin_unlock_irqrestore(&port->spin_lock,flags);
	//SLTTY_DBG("sltty_read_count: %d.\n", ret);
	return ret;
}

int sltty_read_size(struct sltty_port *port)
{
	if (sltty_paranoia_check(port,"sltty_read_size"))
		return -1;
	return TTY_FLIPBUF_SIZE;
}

int sltty_write_count(struct sltty_port *port)
{
	if (sltty_paranoia_check(port,"sltty_write_count"))
		return -1;
	return port->xmit_cnt;
}

int sltty_write_size(struct sltty_port *port)
{
	if (sltty_paranoia_check(port,"sltty_write_size"))
		return -1;
	return port->xmit_size;
}

int sltty_write_flush(struct sltty_port *port)
{
	unsigned long flags;
	SLTTY_DBG("sltty_write_flush...\n");
	if (sltty_paranoia_check(port,"sltty_write_flush"))
		return -1;
	spin_lock_irqsave(&port->spin_lock,flags);
	port->xmit_cnt = port->xmit_head = port->xmit_tail = 0;
	spin_unlock_irqrestore(&port->spin_lock,flags);
	return 0;
}


/* transmitter */
int sltty_tx(struct sltty_port *port, char *buf, int count)
{
	unsigned long flags;
	int cnt;
	int total = 0;

	//SLTTY_DBG("sltty_tx...\n");
	if (sltty_paranoia_check(port,"sltty_tx"))
		return -1;
 
	while (count) {
		cnt = count;
		spin_lock_irqsave(&port->spin_lock, flags);
		if (cnt > port->xmit_cnt)
			cnt = port->xmit_cnt;
		if (cnt > port->xmit_size - port->xmit_tail)
			cnt = port->xmit_size - port->xmit_tail;
		if (cnt <= 0) {
			spin_unlock_irqrestore(&port->spin_lock, flags);
			break;
		}
		memcpy(buf, port->xmit_buf + port->xmit_tail, cnt);
		port->xmit_cnt -= cnt;
		port->xmit_tail = (port->xmit_tail+cnt)%port->xmit_size;
		spin_unlock_irqrestore(&port->spin_lock, flags);
		buf += cnt;
		count -= cnt;
		total += cnt;
	}
	sltty_write_wakeup(port);
	//SLTTY_DBG("sltty_tx: get %d dytes.\n", total);
	return total;
}


/* reciever */
int sltty_rx(struct sltty_port *port, char *buf, int count)
{
	unsigned long flags;
	int total = 0;

	//SLTTY_DBG("sltty_rx...\n");
	if (sltty_paranoia_check(port,"sltty_rx"))
		return -1;

	if(test_bit(TTY_THROTTLED, &port->tty->flags))
		return 0;

       spin_lock_irqsave(&port->spin_lock,flags);
        while (count-- && port->tty->flip.count < TTY_FLIPBUF_SIZE) {
                tty_insert_flip_char(port->tty,*buf++,0);
		total++;
        }
        if (port->tty->flip.count > 0)
                tty_flip_buffer_push(port->tty);
        spin_unlock_irqrestore(&port->spin_lock,flags);
	//SLTTY_DBG("sltty_rx: put %d.\n", total);
	return total;
}


/*
 *	SLTTY Driver specific routines ...
 *
 */

static void do_sltty_modem(void *data)
{
	struct sltty_port *port = data;
	//SLTTY_DBG("sltty_do_modem...\n");
	if(sltty_paranoia_check(port, "do_sltty_modem"))
		return;
	if (!port->modem
	    || port->tty->stopped || port->tty->hw_stopped)
		return;
	modem_run(port->modem);
}

static void sltty_write_wakeup(struct sltty_port *port)
{
	struct tty_struct *tty;
	//SLTTY_DBG("sltty_write_wakeup...\n");
	if(sltty_paranoia_check(port,"sltty_write_wakeup"))
		return;
	tty = port->tty;
	if ((tty->flags & (1 << TTY_DO_WRITE_WAKEUP)) &&
	    tty->ldisc.write_wakeup)
		(tty->ldisc.write_wakeup) (tty);
	wake_up_interruptible(&tty->write_wait);
}

/*
 * sltty_open and friends
 */

static int sltty_setup_port(struct sltty_port *port)
{
	unsigned long page = 0;

	SLTTY_DBG("sltty_setup_port...\n");
	if (port->initialized)
		return 0;
	if (!port->xmit_size) {
		page = __get_free_page(GFP_KERNEL);
		if (!page)
			return -ENOMEM;
	}
	spin_lock(&port->spin_lock);
	if (port->xmit_size) {
		spin_unlock(&port->spin_lock);
		free_page(page);
		return -ERESTARTSYS;
	}
	port->xmit_buf = (char *)page;
	port->xmit_size = PAGE_SIZE;
	port->xmit_cnt = port->xmit_head = port->xmit_tail = 0;
	spin_unlock(&port->spin_lock);

	if (port->tty)
		clear_bit(TTY_IO_ERROR, &port->tty->flags);
	/* port->status = TIOCM_DTR|TIOCM_RTS; */
	if (modem_open(port->modem)) {
		port->xmit_size = 0;
		port->xmit_buf =  0;
		free_page(page);
		return -ENODEV;
	}
	port->initialized = 1;
	return 0;
}

static int block_til_ready(struct tty_struct *tty, struct file *filp,
			   struct sltty_port *port)
{
	int do_clocal = 0, retval;
	DECLARE_WAITQUEUE(wait, current);

	/* block if port is in the process of being closed */
	SLTTY_DBG("block_til_ready...\n");

	if (tty_hung_up_p(filp) || tty->closing) {
		interruptible_sleep_on(&port->close_wait);
		return -ERESTARTSYS;
	}

	/* if non-blocking mode is set ... */
	if ((filp->f_flags & O_NONBLOCK)
	    || (tty->flags & (1 << TTY_IO_ERROR))) {
		return 0;
	}

	/* block waiting for DCD */
	do_clocal = C_CLOCAL(tty);
	retval = 0;
	add_wait_queue(&port->open_wait, &wait);
	cli();
	if (!tty_hung_up_p(filp))
		port->count--;
	sti();
	port->blocked_open++;

	SLTTY_DBG("block_til_ready: waiting for DCD...\n");
	while (1) {
		set_current_state(TASK_INTERRUPTIBLE);
		if (tty_hung_up_p(filp) || !port->initialized) {
			retval = -ERESTARTSYS;
			break;
		}
		if (!tty->closing &&
		    (do_clocal || (port->status & TIOCM_CD))) {
			break;
		}
		if (signal_pending(current)) {
			retval = -ERESTARTSYS;
			break;
		}
		schedule();
	}
	current->state = TASK_RUNNING;
	remove_wait_queue(&port->open_wait, &wait);
	if (!tty_hung_up_p(filp))
		port->count++;
	port->blocked_open--;
	return retval;
}


static int sltty_open(struct tty_struct *tty, struct file *filp)
{
	struct sltty_port *port;
	unsigned int line;
	int error = 0;

	SLTTY_DBG("sltty_open...\n");

	MOD_INC_USE_COUNT;
	line = MINOR(tty->device) - tty->driver.minor_start;
	if (line != 0)
		return -ENODEV;

	port = &sltty_port;

	port->count++;
	tty->driver_data = port;
	port->tty = tty;
	if ((error = sltty_setup_port(port)) != 0)
		return error;

	if ((error = block_til_ready(tty, filp, port)) != 0)
		return error;

	SLTTY_DBG("sltty_open: done.\n");
	return 0;
}

/* close et all */
static void sltty_shutdown_port(struct sltty_port *port)
{
	SLTTY_DBG("sltty_shutdown_port...\n");
	if(sltty_paranoia_check(port,"sltty_shutdown_port"))
		return;
	if (!port->initialized || port->closing)
		return;
	port->closing = 1;

	if (port->tty)
		set_bit(TTY_IO_ERROR, &port->tty->flags);

	modem_close(port->modem);

	if (port->xmit_size) {
		unsigned long page = (unsigned long)port->xmit_buf;
		port->xmit_size = 0;
		port->xmit_head = port->xmit_tail = port->xmit_cnt = 0;
		port->xmit_buf = 0;
		free_page(page);
	}
        if (!port->tty || C_HUPCL(port->tty)) 
                /* drop dtr on this port */
                port->status &= ~TIOCM_DTR;
	port->initialized = 0;
	port->closing = 0;
}

static void sltty_close(struct tty_struct *tty, struct file *filp)
{
	struct sltty_port *port = (struct sltty_port *) tty->driver_data;

	SLTTY_DBG("sltty_close...\n");

	if (tty_hung_up_p(filp)) {
		goto finish;
	}
#if SLTTY_PARANOIA_CHECK
	if ((tty->count == 1) && (port->count != 1)) {
		SLTTY_ERR("sltty_close: bad port count %d.\n",port->count);
		port->count = 1;
	}
#endif

	if (--port->count)
		goto finish;

	tty->closing = 1;
	sltty_shutdown_port(port);

	tty->closing = 0;
	port->tty = 0;
	if (port->blocked_open)
		wake_up_interruptible(&port->open_wait);
	wake_up_interruptible(&port->close_wait);
      finish:
	MOD_DEC_USE_COUNT;
	SLTTY_DBG("sltty_close: done.\n");
}


/* hangup et all */
static void sltty_hangup(struct tty_struct *tty)
{
	struct sltty_port *port = (struct sltty_port *) tty->driver_data;
	SLTTY_DBG("sltty_hangup...\n");
	if (sltty_paranoia_check(port,"sltty_hangup"))
		return;
	sltty_shutdown_port(port);
	port->count = 0;
	port->tty = 0;
	wake_up_interruptible(&port->open_wait);
}

/* write et all */
static int sltty_write(struct tty_struct *tty, int from_user,
		     const unsigned char *buf, int count)
{
	struct sltty_port *port = (struct sltty_port *) tty->driver_data;
	unsigned long flags;
	int cnt, total = 0;

	//SLTTY_DBG("sltty write (from %s): %d...\n",
	//	  from_user?"user":"kern",count);
	if (sltty_paranoia_check(port,"sltty_write"))
		return 0;

	if (!port->xmit_size || !tmp_buf)
		return 0;
	if (from_user)
		down(&tmp_buf_sem);  /* acquire xclusive access to tmp_buf */

	while(count) {
		char *b;
		cnt = count;
		spin_lock_irqsave(&port->spin_lock, flags);
		cnt = MIN(cnt,MIN(port->xmit_size - port->xmit_cnt,
				  port->xmit_size - port->xmit_head));
		if ( cnt <= 0 ) {
			spin_unlock_irqrestore(&port->spin_lock, flags);
			break;
		}
		if (from_user) {
			/* the following may block for paging... hence 
			   enabling interrupts but tx routine may have 
			   created more space in xmit_buf when the ctrl 
			   gets back here  */
			sti();
			copy_from_user(tmp_buf, buf, cnt);
			cli();
			cnt = MIN(cnt,
				  MIN(port->xmit_size - port->xmit_cnt,
				      port->xmit_size - port->xmit_head));
			b = (char *)tmp_buf;
		}
		else {
			b = (char *)buf;
		}

		memcpy(port->xmit_buf + port->xmit_head, b, cnt);
		port->xmit_cnt += cnt;
		port->xmit_head = (port->xmit_head+cnt)%port->xmit_size;
		spin_unlock_irqrestore(&port->spin_lock, flags);
		buf += cnt;
		count -= cnt;
		total += cnt;
#if SLTTY_PARANOIA_PRINT
		{
			int i;
			for (i = 0; i < cnt ; i++)
				SLTTY_DBG("sltty_write: `%c'\n",
					  isprint(b[i])?b[i]:'x');
		}
#endif
	}
	if (from_user)
		up(&tmp_buf_sem);

	if(!tty->stopped && !tty->hw_stopped)
		schedule_task(&port->modem_task);

	//SLTTY_DBG("sltty_write: written %d.\n", total);
	return total;
}

/* flush_chars et all */
static void sltty_flush_chars(struct tty_struct *tty)
{
	struct sltty_port *port = (struct sltty_port *) tty->driver_data;
	SLTTY_DBG("sltty_flush_chars...\n");
	if (sltty_paranoia_check(port,"sltty_flush_chars"))
		return;
	if (port->xmit_cnt <= 0 || tty->stopped || tty->hw_stopped
	    || !port->xmit_size)
		return;
	schedule_task(&port->modem_task);
}

/* write_room */
static int sltty_write_room(struct tty_struct *tty)
{
	struct sltty_port *port;
	unsigned long flags;
	int free;
	//SLTTY_DBG("sltty_write_room...\n");
	if (!tty) {
		SLTTY_ERR("sltty_write_room: invalid arg.\n");
		return -1;
	}
	port = (struct sltty_port *) tty->driver_data;
	if (sltty_paranoia_check(port,"sltty_write_room"))
		return -1;

	spin_lock_irqsave(&port->spin_lock,flags);
	free = port->xmit_size - port->xmit_cnt;
	spin_unlock_irqrestore(&port->spin_lock,flags);
	return free;
}

/* chars_in_buffer et all */
static int sltty_chars_in_buffer(struct tty_struct *tty)
{
	struct sltty_port *port = (struct sltty_port *) tty->driver_data;
	unsigned long flags;
	int ret;
	//SLTTY_DBG("sltty_chars_in_buffer...\n");
	if (sltty_paranoia_check(port,"sltty_chars_in_buffer"))
		return -1;
	spin_lock_irqsave(&port->spin_lock,flags);
	ret = port->xmit_cnt;
	spin_unlock_irqrestore(&port->spin_lock,flags);
	return ret;
}

/* flush_buffer et all */
static void sltty_flush_buffer(struct tty_struct *tty)
{
	struct sltty_port *port = (struct sltty_port *) tty->driver_data;
	SLTTY_DBG("sltty_flush_buffer...\n");
	if (sltty_paranoia_check(port,"sltty_flush_buffer"))
		return;
	sltty_write_flush(port);
	sltty_write_wakeup(port);
}

/* break emulation */
static void sltty_break_ctl(struct tty_struct *tty, int state)
{
	SLTTY_DBG("sltty_break_ctl...\n");
}

/* ioctl et all */
extern inline int sltty_get_modem_info(struct sltty_port *port, unsigned int *value)
{
	unsigned int info = port->status;
	//SLTTY_DBG("sltty_get_modem_info: %x...\n",info);
	put_user(info, (unsigned long *) value);
	return 0;
}


extern inline int sltty_set_modem_info(struct sltty_port *port, unsigned int cmd,
			      unsigned int *value)
{
	unsigned int arg = 0;
	unsigned long flags;

	SLTTY_DBG("sltty_set_modem_info: cmd %x...\n", cmd);

	if (get_user(arg, value))
		return -EFAULT;

	spin_lock_irqsave(&port->spin_lock, flags);
	switch (cmd) {
	case TIOCMBIS:
		port->status |= arg & (TIOCM_RTS|TIOCM_DTR);
		break;

	case TIOCMBIC:
		port->status &= ~ (arg & (TIOCM_RTS|TIOCM_DTR));
		break;

	case TIOCMSET:
		port->status &= ~(TIOCM_RTS|TIOCM_DTR);
		port->status |= arg & (TIOCM_RTS|TIOCM_DTR);
		break;

	default:
		spin_unlock_irqrestore(&port->spin_lock, flags);
		return -EINVAL;
	}
	spin_unlock_irqrestore(&port->spin_lock, flags);
	return 0;
}


static int sltty_ioctl(struct tty_struct *tty, struct file *filp,
		     unsigned int cmd, unsigned long arg)
{
	struct sltty_port *port = (struct sltty_port *)tty->driver_data;
	//SLTTY_DBG("sltty_ioctl: cmd %x...\n", cmd);
	if (sltty_paranoia_check(port,"sltty_ioctl"))
		return -ENODEV;

	switch (cmd) {
	case TIOCMGET:
		return sltty_get_modem_info(port, (unsigned int *) arg);
	case TIOCMBIS:
	case TIOCMBIC:
	case TIOCMSET:
		return sltty_set_modem_info(port, cmd, (unsigned int *) arg);
	/* virtual UART is protected by PCTel's patent */
	case TIOCGSERIAL: case TIOCSSERIAL:
	default:
		return -ENOIOCTLCMD;
	}
	return 0;
}

/* set_termios et all */
static void sltty_set_termios(struct tty_struct *tty,
			      struct termios *old_termios)
{
	struct sltty_port *port = (struct sltty_port *) tty->driver_data;
	//SLTTY_DBG("sltty_set_termios...\n");
	if (sltty_paranoia_check(port,"sltty_set_termios"))
		return;
	if(!tty_get_baud_rate(port->tty)) {
		SLTTY_DBG("sltty_set_termios: hangup modem.\n");
		modem_hangup(port->modem);
		return;
	}
}


/*
 * init funcs
 */

static int register_drivers(void)
{
	int error;
	SLTTY_DBG("register_drivers...\n");
	/* tty driver structure initialization */
	memset(&sltty, 0, sizeof(struct tty_driver));
	sltty.magic = TTY_DRIVER_MAGIC;
	sltty.driver_name = "sltty";
	sltty.name = "ttySL";
	sltty.major = SLTTY_NMAJOR;
	sltty.minor_start = 0;
	sltty.num = 1;
	sltty.type = TTY_DRIVER_TYPE_SERIAL;
	sltty.subtype = SERIAL_TYPE_NORMAL;
	sltty.init_termios = tty_std_termios;
	sltty.init_termios.c_cflag =
	    B115200 | CS8 | CREAD | HUPCL | CLOCAL;
	sltty.flags = TTY_DRIVER_REAL_RAW | TTY_DRIVER_NO_DEVFS;
	sltty.refcount = &sltty_refcount;

	sltty.table = &sltty_table;
	sltty.termios = &sltty_termios;
	sltty.termios_locked = &sltty_termios_locked;

	sltty.open = sltty_open;
	sltty.close = sltty_close;
	sltty.write = sltty_write;
	sltty.put_char = NULL /*sltty_put_char */ ;
	sltty.flush_chars = sltty_flush_chars;
	sltty.write_room = sltty_write_room;
	sltty.chars_in_buffer = sltty_chars_in_buffer;
	sltty.ioctl = sltty_ioctl;
	sltty.set_termios = sltty_set_termios;
	sltty.break_ctl   = sltty_break_ctl;
	sltty.hangup = sltty_hangup;
	sltty.flush_buffer = sltty_flush_buffer;

	if ((error = tty_register_driver(&sltty)) != 0) {
		SLTTY_ERR("sltty: couldn't register the tty driver, err=%d\n",
			  error);
		return error;
	}
	return 0;
}

static void unregister_drivers(void)
{
	int error;
	error = tty_unregister_driver(&sltty);
	if(error)
		SLTTY_ERR("couldn't unregister tty driver error=%d.\n",
			  error);
}



int sltty_init(void)
{
	struct sltty_port *port;
	unsigned long page;
	int rc;

	SLTTY_DBG("sltty_init...\n");

	if (!tmp_buf) {
		page = __get_free_page(GFP_KERNEL);
		if (!page) {
			SLTTY_ERR("sltty_init: out of mem.\n");
			return -ENOMEM;
		}
		tmp_buf = (unsigned char *) page;
	}

	memset(&sltty_port, 0, sizeof(sltty_port));
	port = &sltty_port;

	port->magic = SLTTY_MAGIC;
	INIT_TQUEUE(&port->modem_task,do_sltty_modem,port);
	port->status = 0;
	init_waitqueue_head(&port->close_wait);
	init_waitqueue_head(&port->open_wait);

	port->modem = modem_create(port);
	if (!port->modem) {
		free_page((unsigned long)tmp_buf);
		return -ENODEV;
	}
	rc = register_drivers();
	if (rc) {
		free_page((unsigned long)tmp_buf);
		return rc;
	}

	return 0;
}


void sltty_exit(void)
{
	void *modem;
	SLTTY_DBG("sltty_exit...\n");
	unregister_drivers();
	modem = sltty_port.modem;
	sltty_port.modem = 0;
	modem_delete(modem);
	free_pages((unsigned long) tmp_buf, 0);
}


